#!/usr/bin/env python
# coding: utf-8

# In[1]:


import pandas as pd
import numpy as np
from sklearn.linear_model import ElasticNet
from sklearn.preprocessing import StandardScaler
from sklearn.impute import SimpleImputer


# ### README

# Este código contiene la ejecución de la predicción para el problema de producción del antígeno propuesto en la datathon UNIVERSITYHACK 2024.

# Autores: Ana Porras Garrido y Pedro José Lucas Guillén - Equipo Named17 

# Es necesario definir la variable FOLDER como la ruta en la que se ubican los archivos del reto. Se utilizan todos los archivos proporcionados en sus últimas versiones.

# En requirements se listan la versión de Python y las de los paquetes utilizados.

# #### REQUIREMENTS

# Python 3.12.5

# * anyio==4.6.0
# * argon2-cffi==23.1.0
# * argon2-cffi-bindings==21.2.0
# * arrow==1.3.0
# * asttokens==2.4.1
# * async-lru==2.0.4
# * attrs==24.2.0
# * babel==2.16.0
# * beautifulsoup4==4.12.3
# * bleach==6.1.0
# * certifi==2024.8.30
# * cffi==1.17.1
# * charset-normalizer==3.3.2
# * colorama==0.4.6
# * comm==0.2.2
# * contourpy==1.3.0
# * cycler==0.12.1
# * debugpy==1.8.5
# * decorator==5.1.1
# * defusedxml==0.7.1
# * et-xmlfile==1.1.0
# * executing==2.1.0
# * fastjsonschema==2.20.0
# * fonttools==4.54.0
# * fqdn==1.5.1
# * h11==0.14.0
# * httpcore==1.0.5
# * httpx==0.27.2
# * idna==3.10
# * ipykernel==6.29.5
# * ipython==8.27.0
# * isoduration==20.11.0
# * jedi==0.19.1
# * Jinja2==3.1.4
# * joblib==1.4.2
# * json5==0.9.25
# * jsonpointer==3.0.0
# * jsonschema==4.23.0
# * jsonschema-specifications==2023.12.1
# * jupyter-events==0.10.0
# * jupyter-lsp==2.2.5
# * jupyter_client==8.6.3
# * jupyter_core==5.7.2
# * jupyter_server==2.14.2
# * jupyter_server_terminals==0.5.3
# * jupyterlab==4.2.5
# * jupyterlab_pygments==0.3.0
# * jupyterlab_server==2.27.3
# * kiwisolver==1.4.7
# * lightgbm==4.5.0
# * MarkupSafe==2.1.5
# * matplotlib==3.9.2
# * matplotlib-inline==0.1.7
# * mistune==3.0.2
# * nbclient==0.10.0
# * nbconvert==7.16.4
# * nbformat==5.10.4
# * nest-asyncio==1.6.0
# * notebook_shim==0.2.4
# * numpy==2.1.1
# * openpyxl==3.1.5
# * overrides==7.7.0
# * packaging==24.1
# * pandas==2.2.3
# * pandocfilters==1.5.1
# * parso==0.8.4
# * patsy==0.5.6
# * pillow==10.4.0
# * platformdirs==4.3.6
# * prometheus_client==0.21.0
# * prompt_toolkit==3.0.47
# * psutil==6.0.0
# * pure_eval==0.2.3
# * pycparser==2.22
# * Pygments==2.18.0
# * pyparsing==3.1.4
# * python-dateutil==2.9.0.post0
# * python-json-logger==2.0.7
# * pytz==2024.2
# * pywin32==306
# * pywinpty==2.0.13
# * PyYAML==6.0.2
# * pyzmq==26.2.0
# * referencing==0.35.1
# * requests==2.32.3
# * rfc3339-validator==0.1.4
# * rfc3986-validator==0.1.1
# * rpds-py==0.20.0
# * scikit-learn==1.5.2
# * scipy==1.14.1
# * seaborn==0.13.2
# * Send2Trash==1.8.3
# * setuptools==75.1.0
# * six==1.16.0
# * sniffio==1.3.1
# * soupsieve==2.6
# * stack-data==0.6.3
# * statsmodels==0.14.4
# * terminado==0.18.1
# * threadpoolctl==3.5.0
# * tinycss2==1.3.0
# * tornado==6.4.1
# * tqdm==4.66.5
# * traitlets==5.14.3
# * types-python-dateutil==2.9.0.20240906
# * tzdata==2024.2
# * uri-template==1.3.0
# * urllib3==2.2.3
# * wcwidth==0.2.13
# * webcolors==24.8.0
# * webencodings==0.5.1
# * websocket-client==1.8.0
# * wheel==0.44.0
# * xgboost==2.1.1

# ### PARAMETERS

# In[2]:


FOLDER = '20240927 Reto UniversityHack 2024/'
FILES = {
    'producción': ['Fases producción v03.xlsx'],
    'biorreactor': ['Biorreactor 13169.xlsx', 'Biorreactor 13170.xlsx', 'Biorreactor 13171.xlsx', 
                    'Biorreactor 13172.xlsx', 'Biorreactor 14614.xlsx', 'Biorreactor 14615.xlsx', 
                    'Biorreactor 14616.xlsx', 'Biorreactor 14617.xlsx', 'Biorreactor 14618.xlsx'],
    'cinéticos': ['Cinéticos IPC.xlsx'],
    'componentes': ['Movimientos componentes.xlsx'],
    'ambientales': ['Temperaturas y humedades.xlsx'],
    'test': ['Fases producción v03 Test.xlsx'],
}

biorreactores_id = [int(x.replace('Biorreactor ', '').replace('.xlsx', '')) for x in FILES['biorreactor']]


# ### FUNCTIONS

# In[3]:


def preprocess(var, drop_unique=True):
    var = var.drop_duplicates()
    if drop_unique:
        to_drop = var.columns[var.nunique() == 1]
        var = var.drop(to_drop, axis=1)
    parse_dates = var.dtypes[var.dtypes == 'datetime64[ns]'].index
    # fix number formats
    var = var.replace({float('nan'): None, 'N.A': None, 'NA': None, 'na': None, '-': None, 'LO': None, 'NR': None})
    for col in var.columns:
        if 'mixed' in pd.api.types.infer_dtype(var[col]):
            nona = ~var[col].isna()
            try: 
                var.loc[nona, col] = var.loc[nona, col].astype(str).str.replace(',', '.').str.replace('+', '').astype(float)
            except:
                pass
    for col in var.columns:
        if 'mixed' in pd.api.types.infer_dtype(var[col]):
            nona = ~var[col].isna()
            try: 
                var.loc[nona, col] = var.loc[nona, col].astype(str).str.replace(',', '.').str.split('/').str[0].astype(float)
            except:
                pass
    for col in parse_dates:
        var[col] = pd.to_datetime(var[col])
    return var

def biorreactor_lotes(var, key):
    return var[var['ID Bioreactor'] == key].sort_values(by='Fecha/hora inicio')

def buscar_lote_previo(var, lote):
    b = biorreactor_lotes(var, var.loc[lote, 'ID Bioreactor'])
    b['lote_previo'] = b.index
    b['lote_previo'] = b['lote_previo'].shift(1)
    return b.loc[lote, 'lote_previo']

def check_peso(fin_parental, ini_has_parental, bio_id):
    b = biorreactores[bio_id]
    b = b[(b.index > fin_parental) & (b.index < ini_has_parental)]
    b = b['Load_Cell_Net_PV']
    b = b.clip(lower=0)
    peso_minimo = b.min()
    if len(b) > 0:
        peso_reduccion = peso_minimo / max(b.loc[:b.idxmin()].max(), 1) - 1
    else:
        # suponemos que el primero es independiente
        peso_reduccion = 0
    return peso_minimo, peso_reduccion

def check_parental_previo(var, aux=None):
    if aux is None:
        aux = var.copy()
        
    cols = ['ID Bioreactor', 'Orden en el encadenado', 'Fecha/hora inicio', 'Fecha/hora fin']
    parental = var[cols + ['LOTE parental']].merge(var[cols], how='left', left_on='LOTE parental', right_index=True, suffixes=('', '_parental'))
    parental['lag'] = parental['Fecha/hora inicio'] - parental['Fecha/hora fin_parental']
    peso = {lote: check_peso(parental.loc[lote, 'Fecha/hora fin_parental'], parental.loc[lote, 'Fecha/hora inicio'], parental.loc[lote, 'ID Bioreactor']) for lote in var.index if var.loc[lote, 'LOTE parental']}
    peso = pd.DataFrame.from_dict(peso, orient='index', columns=['peso_minimo', 'peso_reduccion'])
    peso = peso.reindex(var.index)
    parental = parental.join(peso)
    peso_p = {lote: check_peso(parental.loc[lote, 'Fecha/hora fin_parental'], parental.loc[lote, 'Fecha/hora inicio'], parental.loc[lote, 'ID Bioreactor_parental']) for lote in var.index if var.loc[lote, 'LOTE parental']}
    peso_p = pd.DataFrame.from_dict(peso_p, orient='index', columns=['peso_minimo', 'peso_reduccion'])
    peso_p = peso_p.reindex(var.index)
    parental = parental.join(peso_p, lsuffix='_bio', rsuffix='_bio_parental')
    
    previo = {lote: buscar_lote_previo(aux, lote) for lote in var.index}
    var['LOTE previo'] = pd.Series(previo)
    previo = var[cols + ['LOTE previo']].merge(aux[cols], how='left', left_on='LOTE previo', right_index=True, suffixes=('', '_previo'))
    previo['lag'] = previo['Fecha/hora inicio'] - previo['Fecha/hora fin_previo']
    peso_pr = {lote: check_peso(previo.loc[lote, 'Fecha/hora fin_previo'], previo.loc[lote, 'Fecha/hora inicio'], previo.loc[lote, 'ID Bioreactor']) for lote in var.index if var.loc[lote, 'LOTE previo']}
    peso_pr = pd.DataFrame.from_dict(peso_pr, orient='index', columns=['peso_minimo', 'peso_reduccion'])
    peso_pr = peso_pr.reindex(var.index)
    previo = previo.join(peso_pr)

    return parental, previo


def agg_cineticos(cineticos, group_by=['LOTE']):
    first = cineticos.groupby(group_by).first(skipna=False)
    last = cineticos.groupby(group_by).last(skipna=False)
    mean = cineticos.groupby(group_by).mean(numeric_only=True) 
    # quito std porque el número de datos en test es en general menor y eso puede hacerlo no comparable
    # podría sesgar el modelo
    # std = cineticos.groupby(group_by).std(numeric_only=True) 
    var = np.log(last[mean.columns] / first[mean.columns])
    var[first[mean.columns] == 0] = float('nan')
    count = cineticos.reset_index().groupby(group_by).count()['index'].rename('count')
    mean[count == 1] = float('nan')
    first[count == 1] = float('nan')
    agg = first.join(last, lsuffix='_ini', rsuffix='_fin')
    agg = agg.join(mean).join(var, rsuffix='_var')
    agg = agg.join(count)
    return agg

def agg_temporal(data, lote, id_, fecha_ini, fecha_fin):
    b = data[id_].loc[fecha_ini:fecha_fin]
    b = b.astype(float)
    r = b.diff()
    agg = [b.mean(), b.std(), r.mean(), r.std()] # median, min, max
    agg = [pd.DataFrame(x.rename(lote)).T for x in agg]
    agg = agg[0].join(agg[1], rsuffix='_std').join(agg[2], rsuffix='_ret').join(agg[3], rsuffix='_ret_std')
    return agg

def impute_nan(feats, impute_kind, t):    
    imputed_features = feats.copy() 
    imputer = SimpleImputer(strategy=impute_kind.lower())
    imputer.fit(feats.drop(t));
    imputed_features = imputer.transform(feats)
    imputed_features = pd.DataFrame(imputed_features, index=feats.index, columns=feats.columns)
    return imputed_features


# ### READ & PREPROCESS

# ##### fases

# In[4]:


inoculo = pd.read_excel(FOLDER + FILES['producción'][0], sheet_name='Inóculo', parse_dates=['Fecha/hora inicio', 'Fecha/hora fin'])
inoculo = preprocess(inoculo)
inoculo['LOTE'] = inoculo['LOTE'].astype(str)
inoculo = inoculo.set_index('LOTE')
inoculo['ID bioreactor'] = inoculo['ID bioreactor'].astype(float).round(0).astype(int)


# In[5]:


cultivo = pd.read_excel(FOLDER + FILES['producción'][0], sheet_name='Cultivo final', parse_dates=['Fecha/hora inicio', 'Fecha/hora fin'])
cultivo = preprocess(cultivo)
cultivo['LOTE'] = cultivo['LOTE'].astype(float).round(0).astype(int).astype(str)
nona = ~cultivo['LOTE parental'].isna()
cultivo.loc[nona, 'LOTE parental'] = cultivo.loc[nona, 'LOTE parental'].astype(float).round(0).astype(int).astype(str)
cultivo = cultivo.set_index('LOTE')
cultivo['ID Bioreactor'] = cultivo['ID Bioreactor'].round(0).astype(int)

to_remove = ['23052', '23053', '23066', '23074', '24002', '24029', '24043'] + ['24026', '24034', '24039', '24048'] + ['24011', '24015']
cultivo = cultivo.drop(to_remove)
cultivo.loc['24017', 'ID Bioreactor'] = cultivo.loc[cultivo.loc['24017', 'LOTE parental'], 'ID Bioreactor']


# In[6]:


test = pd.read_excel(FOLDER + FILES['test'][0], sheet_name='Cultivo final', parse_dates=['Fecha/hora inicio', 'Fecha/hora fin'])#, dtype={'ID Centrífuga': float})
test = preprocess(test)
test['LOTE'] = test['LOTE'].astype(str)
test.loc[~test['LOTE parental'].isna(), 'LOTE parental'] = test.loc[~test['LOTE parental'].isna(), 'LOTE parental'].round(0).astype(int).astype(str)
test = test.set_index('LOTE')
test['ID Bioreactor'] = test['ID Bioreactor'].astype(float).round(0).astype(int)


# ##### biorreactores

# In[7]:


biorreactores = {}
cols = ['Load_Cell_Net_PV', 'PUMP_2_PV', 'Air_Sparge_PV', 'Gas_Overlay_PV']
for file, bio_id in zip(FILES['biorreactor'], biorreactores_id):
    aux = pd.read_excel(FOLDER + file, sheet_name='Datos', header=[0], parse_dates=['DateTime'])
    aux = preprocess(aux, drop_unique=False)
    aux.columns = [x.replace(str(bio_id) + '_', '').replace('FERM0101.', '') for x in aux.columns]
    aux = aux.set_index('DateTime').sort_index()
    biorreactores[bio_id] = aux[cols]


# ##### dependencias

# In[8]:


var = pd.concat([cultivo, test])
var['ID Bioreactor_13170'] = (var['ID Bioreactor'] == 13170).astype(int)

parental, previo = check_parental_previo(var)

dependent = var[~var['LOTE parental'].isna()].index

dependencias = parental.dropna(subset='LOTE parental').set_index('LOTE parental')['peso_minimo_bio']
dependencias = dependencias.rename('peso_siguiente').reindex(var.index).fillna(0)

# enganche de los parentales: añadimos nuevos datos de preinóculo e inóculo al conjunto inicial
orden_2 = var.loc[var.index.isin(dependent) & (var['Orden en el encadenado'] == 2)].index
rename = var.loc[orden_2, 'LOTE parental'].reset_index().set_index('LOTE parental')['LOTE']
inoculo_ = inoculo.loc[rename.index].rename(rename)
inoculo = pd.concat([inoculo, inoculo_])

orden_3 = var.loc[var.index.isin(dependent) & (var['Orden en el encadenado'] == 3)].index
rename = var.loc[orden_3, 'LOTE parental'].reset_index().set_index('LOTE parental')['LOTE']
inoculo_ = inoculo.loc[rename.index].rename(rename)
inoculo = pd.concat([inoculo, inoculo_])


# ##### cinéticos

# In[9]:


cultivo_cin = pd.read_excel(FOLDER + FILES['cinéticos'][0], sheet_name='Cultivos finales', parse_dates=['Fecha'])
cultivo_cin = preprocess(cultivo_cin)
cultivo_cin = cultivo_cin.rename({'Lote': 'LOTE'}, axis=1)
cultivo_cin['LOTE'] = cultivo_cin['LOTE'].astype(str)
float_cols = ['Turbidez', 'Viabilidad', 'Glucosa g/L']
cultivo_cin[float_cols] = cultivo_cin[float_cols].astype(float)

cultivo_cin = agg_cineticos(cultivo_cin)


# In[10]:


pairs = [['Fecha/hora inicio', 'Fecha_ini'], 
         ['Fecha/hora fin', 'Fecha_fin'], 
         ['Turbidez fin cultivo', 'Turbidez_fin'],]

for x, y in pairs:
    var[x] = var[x].fillna(cultivo_cin[y])
    cultivo_cin[y] = cultivo_cin[y].fillna(var[x])


# In[11]:


var = var.join(cultivo_cin)


# ##### datos biorreactores

# In[12]:


bio_inoculo = [agg_temporal(biorreactores, lote, inoculo.loc[lote, 'ID bioreactor'], 
                            inoculo.loc[lote, 'Fecha/hora inicio'], inoculo.loc[lote, 'Fecha/hora fin']) 
               for lote in var.index if inoculo.loc[lote, 'ID bioreactor'] in biorreactores_id]
bio_inoculo = pd.concat(bio_inoculo)


# In[13]:


bio_cultivo = [agg_temporal(biorreactores, lote, var.loc[lote, 'ID Bioreactor'], 
                            var.loc[lote, 'Fecha/hora inicio'], var.loc[lote, 'Fecha/hora fin']) 
               for lote in var.index if var.loc[lote, 'ID Bioreactor'] in biorreactores_id]
bio_cultivo = pd.concat(bio_cultivo)


# ##### materiales

# In[14]:


material = pd.read_excel(FOLDER + FILES['componentes'][0], sheet_name='Full1', parse_dates=['Fecha recepción', 'Fecha traslado'])

material = material.rename({'Lote': 'LOTE'}, axis=1)
material['LOTE'] = material['LOTE'].astype(str)

materiales_id = [100005, 100012, 100004, 100008]
material = material[material['Material'].isin(materiales_id)]


# In[15]:


material = material.groupby(['LOTE', 'Material', 'Lote interno', 'Lote Proveedor', 'Fecha recepción', 'Fecha traslado']).sum().reset_index()
material = material[material['Qty'] > 0]
total = material.groupby(['LOTE', 'Material']).sum(numeric_only=True)['Qty'].rename('total').reset_index()
material = material.merge(total, how='left', on=['LOTE', 'Material'])
material['proporcion'] = material['Qty'] / material['total']
material['tiempo_almacén'] = (material['Fecha traslado'] - material['Fecha recepción']).dt.days

material = material.merge(var[['Fecha/hora inicio', 'Fecha/hora fin']], how='left', left_on='LOTE', right_index=True)
material = material[material['LOTE'].isin(var.index)]

material['lag_traslado'] = (material['Fecha/hora fin'] - material['Fecha traslado']).dt.days

to_fix = material[material['lag_traslado'] < 0].index
cols = [ 'Material', 'Lote interno', 'Lote Proveedor', 'Fecha recepción']
for i in to_fix:
    aux = pd.DataFrame([material.loc[i, cols].values] * material.shape[0], index=material.index, columns=cols)
    fechas = material[(material[cols] ==aux).all(axis=1)]['Fecha traslado'].value_counts()
    fechas = fechas[fechas.index < material.loc[i, 'Fecha/hora fin']]
    material.loc[i, 'Fecha traslado'] = fechas.index[0]

material['lag_traslado'] = (material['Fecha/hora fin'] - material['Fecha traslado']).dt.days
material['lag_traslado_w'] = material['lag_traslado'] * material['proporcion']

aux = material['Fecha traslado'] + pd.Timedelta(1, 'D')
material['Fecha almacen prod'] = pd.concat([material['Fecha/hora inicio'], aux], axis=1).max(axis=1)

materiales = pd.pivot_table(material, index='LOTE', columns='Material', values='lag_traslado_w', aggfunc='sum').fillna(0)
materiales = pd.DataFrame(materiales[100008].rename('traslado_100008'))


# ##### ambientales

# In[16]:


ambientales = pd.read_excel(FOLDER + FILES['ambientales'][0], sheet_name='Datos', parse_dates=['DateTime'])
nuevos_nombres ={'DateTime':'Fecha', '06299_MI1302.PV':'H_bios', 
                '07633_TI0601.PV':'T_alma_princ', '07781_TI1501.PV':'T_alma_prod'}
ambientales = ambientales.rename(columns=nuevos_nombres)
ambientales = ambientales[nuevos_nombres.values()].set_index('Fecha')


# In[17]:


ambientales_ = {0: ambientales[['H_bios']]}
amb_cultivo = [agg_temporal(ambientales_, lote, 0, var.loc[lote, 'Fecha/hora inicio'], var.loc[lote, 'Fecha/hora fin']) 
                  for lote in var.index]
amb_cultivo = pd.concat(amb_cultivo)

ambientales_ = {0: ambientales[['T_alma_princ']]}
amb_almacen = [agg_temporal(ambientales_, i, 0, material.loc[i, 'Fecha recepción'], material.loc[i, 'Fecha traslado']) 
                for i in material.index]
amb_almacen = pd.concat(amb_almacen)

ambientales_ = {0: ambientales[['T_alma_prod']]}
amb_almacen2 = [agg_temporal(ambientales_, i, 0, material.loc[i, 'Fecha traslado'], material.loc[i, 'Fecha almacen prod']) 
                for i in material.index]
amb_almacen2 = pd.concat(amb_almacen2)

amb_almacen_by_material = []
for col in amb_almacen.columns:
    m = material[~amb_almacen[col].isna()]
    t = m.groupby(['LOTE', 'Material']).sum(numeric_only=True)['Qty'].rename('Total')
    m = m.merge(t, how='left', left_on=['LOTE', 'Material'], right_index=True)
    p = (m['Qty'] / m['Total']).fillna(1)
    j = pd.concat([material[['LOTE', 'Material']], (amb_almacen[col] * p).rename(col)], axis=1)
    j = j[~amb_almacen[col].isna()]
    s = j.groupby(['LOTE', 'Material']).sum()
    v = pd.pivot_table(s, index='LOTE', columns='Material')
    v.columns = [x[0] + '_' + str(x[1]) for x in v.columns]
    amb_almacen_by_material.append(v)

amb_almacen_by_material = pd.concat(amb_almacen_by_material, axis=1)

amb_almacen_2_by_material = []
for col in amb_almacen2.columns:
    m = material[~amb_almacen2[col].isna()]
    t = m.groupby(['LOTE', 'Material']).sum(numeric_only=True)['Qty'].rename('Total')
    m = m.merge(t, how='left', left_on=['LOTE', 'Material'], right_index=True)
    p = (m['Qty'] / m['Total']).fillna(1)
    j = pd.concat([material[['LOTE', 'Material']], (amb_almacen2[col] * p).rename(col)], axis=1)
    j = j[~amb_almacen2[col].isna()]
    s = j.groupby(['LOTE', 'Material']).sum()
    v = pd.pivot_table(s, index='LOTE', columns='Material')
    v.columns = [x[0] + '_' + str(x[1]) for x in v.columns]
    amb_almacen_2_by_material.append(v)

amb_almacen_2_by_material = pd.concat(amb_almacen_2_by_material, axis=1)


# ### FEATURES

# In[18]:


keys = ['cultivo', 'dependencias', 'bioinoculo', 'biocultivo', 
        'materiales', 'amb_almacen_by_material', 'amb_almacen_2_by_material', 'amb_cultivo']
A = [var, dependencias, bio_inoculo, bio_cultivo]
A += [materiales, amb_almacen_by_material, amb_almacen_2_by_material, amb_cultivo]
A = pd.concat(A, axis=1, keys=keys)
A = A.loc[var.index]
A.columns = [x + '_' + y for x, y in A.columns]
viabilidad = [x for x in A.columns if 'viabilidad' in x.lower()]
A[viabilidad] = A[viabilidad] / 10**6 # los números grandes dan problemas
float_cols = A.dtypes[A.dtypes == 'object'].index
A[float_cols] = A[float_cols].astype(float)
A[~np.isfinite(A)] = float('nan')

A.columns = [x.replace('cultivo_ID Bioreactor_', 'biocultivo_ID Bioreactor_') for x in A.columns]


# In[19]:


selection = ['amb_almacen_2_by_material_T_alma_prod_ret_100005',
             'amb_almacen_2_by_material_T_alma_prod_ret_100012',
             'amb_almacen_by_material_T_alma_princ_100004', 
             'amb_cultivo_H_bios_std',
             'biocultivo_ID Bioreactor_13170', 
             'biocultivo_PUMP_2_PV',
             'bioinoculo_Air_Sparge_PV_std', 
             'bioinoculo_Gas_Overlay_PV_ret_std',
             'cultivo_Glucosa g/L_fin', 
             'cultivo_Turbidez fin cultivo',
             'cultivo_Viabilidad_fin', 
             'dependencias_peso_siguiente',
             'materiales_traslado_100008']


# In[20]:


B = A[selection + ['cultivo_Producto 1']]


# In[21]:


impute_kind = 'mean'
t = B[B['cultivo_Producto 1'].isna()].index
C = impute_nan(B.drop(['cultivo_Producto 1'], axis=1), impute_kind, t)


# In[22]:


scaler = StandardScaler()
scaler.fit(C[~B['cultivo_Producto 1'].isna()]);
D = scaler.transform(C)
D = pd.DataFrame(D, index=C.index, columns=C.columns)

# no se estandarizan las binarias
binary_cols = C.columns[C.nunique() == 2]
D[binary_cols] = C[binary_cols]


# ### PREDICT

# In[23]:


lotes = B.reset_index()['LOTE']
label = B.reset_index()['cultivo_Producto 1'].dropna()
features_all = B.reset_index().drop(['cultivo_Producto 1', 'LOTE'], axis=1)
imputed_features_all = C.reset_index().drop(['LOTE'], axis=1)
z_imputed_features_all = D.reset_index().drop(['LOTE'], axis=1)

features = features_all.loc[label.index]
imputed_features = imputed_features_all.loc[label.index]
z_imputed_features = z_imputed_features_all.loc[label.index]


# In[24]:


alpha = 1
l1_ratio = 0.5
random_state = 1717


# In[25]:


X_ = z_imputed_features
test = z_imputed_features_all[X_.columns].loc[z_imputed_features_all.index.difference(label.index)]


# In[26]:


model = ElasticNet(alpha=alpha, l1_ratio=l1_ratio, random_state=random_state)
model.fit(X_, label)
pred_test = model.predict(test)
pred_test = pd.Series(pred_test, index=test.index)
output = pd.concat([lotes.loc[test.index], pred_test], axis=1)


# In[27]:


OUTPUT_FILE = 'Named17_UH2024.txt'
output.to_csv(OUTPUT_FILE, sep='|', index=False, header=False, float_format='%.6f')


# In[ ]:




